<?php
/**
 * Plugin Name: ARIX Pro CSV Product Importer
 * Plugin URI: https://arixnetwork.com/pro-importer
 * Description: Enterprise-grade WooCommerce product importer with advanced column mapping, draft-only enforcement, and modern SaaS-style UI. Handles large files safely with resume capability.
 * Version: 2.1.0
 * Author: ARIXNETWORK
 * Author URI: https://arixnetwork.com
 * License: GPL-3.0+
 * Text Domain: arix-pro-importer
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 8.5
 */

// Security check
defined('ABSPATH') || exit;

final class ARIX_Pro_Importer {
    private static $instance = null;
    public $version = '2.1.0';
    public $plugin_path;
    public $plugin_url;
    public $assets_url;
    public $import_session = [];
    public $capability = 'manage_woocommerce';

    private function __construct() {
        $this->plugin_path = plugin_dir_path(__FILE__);
        $this->plugin_url  = plugin_dir_url(__FILE__);
        $this->assets_url  = $this->plugin_url . 'assets/';
        
        // Load text domain
        add_action('init', [$this, 'load_textdomain']);
        
        // Admin menu and assets
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        
        // AJAX handlers
        add_action('wp_ajax_arix_upload_csv', [$this, 'handle_csv_upload']);
        add_action('wp_ajax_arix_parse_csv_preview', [$this, 'parse_csv_preview']);
        add_action('wp_ajax_arix_save_mapping', [$this, 'save_mapping_template']);
        add_action('wp_ajax_arix_load_mapping', [$this, 'load_mapping_template']);
        add_action('wp_ajax_arix_delete_mapping', [$this, 'delete_mapping_template']);
        add_action('wp_ajax_arix_start_import', [$this, 'start_import_process']);
        add_action('wp_ajax_arix_process_chunk', [$this, 'process_import_chunk']);
        add_action('wp_ajax_arix_resume_import', [$this, 'resume_import']);
        add_action('wp_ajax_arix_cancel_import', [$this, 'cancel_import']);
        
        // Draft enforcement hooks
        add_filter('wp_insert_post_data', [$this, 'enforce_draft_status'], 99, 2);
        add_filter('woocommerce_new_product_data', [$this, 'enforce_draft_on_variations'], 10, 2);
        
        // Initialize session handling
        add_action('admin_init', [$this, 'init_import_session']);
    }

    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function load_textdomain() {
        load_plugin_textdomain('arix-pro-importer', false, basename(dirname(__FILE__)) . '/languages');
    }

    public function register_admin_menu() {
        add_menu_page(
            __('ARIX Pro Importer', 'arix-pro-importer'),
            __('Pro Importer', 'arix-pro-importer'),
            $this->capability,
            'arix-pro-importer',
            [$this, 'render_admin_page'],
            'image/svg+xml;base64,' . base64_encode($this->get_menu_icon_svg()),
            55
        );
    }

    private function get_menu_icon_svg() {
        return '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20"><path fill="#82878c" d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm-8 14H5v-2h6v2zm0-4H5v-2h6v2zm0-4H5V7h6v2zm6 8h-4v-2h4v2zm0-4h-4v-2h4v2zm0-4h-4V7h4v2z"/></svg>';
    }

    public function enqueue_admin_assets($hook) {
        if ('toplevel_page_arix-pro-importer' !== $hook) {
            return;
        }

        // CSS with dark mode support
        wp_enqueue_style(
            'arix-pro-importer-css',
            $this->assets_url . 'css/admin.css',
            [],
            $this->version
        );
        
        // Modern JS with chunked import handling
        wp_enqueue_script(
            'arix-pro-importer-js',
            $this->assets_url . 'js/admin.js',
            ['jquery', 'jquery-ui-sortable', 'wp-i18n'],
            $this->version,
            true
        );
        
        // Localize script with AJAX URL and translations
        wp_localize_script('arix-pro-importer-js', 'arixImporter', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('arix_importer_nonce'),
            'i18n' => [
                'upload_csv' => __('Upload CSV File', 'arix-pro-importer'),
                'mapping' => __('Column Mapping', 'arix-pro-importer'),
                'settings' => __('Import Settings', 'arix-pro-importer'),
                'importing' => __('Importing Products', 'arix-pro-importer'),
                'complete' => __('Import Complete', 'arix-pro-importer'),
                'error' => __('Error', 'arix-pro-importer'),
                'success' => __('Success', 'arix-pro-importer'),
                'processing' => __('Processing...', 'arix-pro-importer'),
                'resume' => __('Resume Import', 'arix-pro-importer'),
                'cancel' => __('Cancel Import', 'arix-pro-importer'),
                'draft_warning' => __('All products will be saved as DRAFTS only. No products will be published.', 'arix-pro-importer'),
                'file_too_large' => __('File exceeds maximum size. Please split your CSV file.', 'arix-pro-importer'),
                'invalid_csv' => __('Invalid CSV file. Please check encoding and structure.', 'arix-pro-importer'),
                'no_templates' => __('No saved mapping templates found.', 'arix-pro-importer'),
                'template_saved' => __('Mapping template saved successfully!', 'arix-pro-importer'),
                'template_deleted' => __('Template deleted successfully.', 'arix-pro-importer'),
                'import_resumed' => __('Import resumed successfully.', 'arix-pro-importer'),
                'import_cancelled' => __('Import cancelled and cleaned up.', 'arix-pro-importer'),
                'chunk_size' => apply_filters('arix_importer_chunk_size', 25),
                'max_file_size' => wp_max_upload_size(),
                'supported_encodings' => ['UTF-8', 'UTF-8-BOM', 'ISO-8859-1', 'Windows-1252']
            ],
            'capabilities' => [
                'can_publish' => current_user_can('publish_products'),
                'is_admin' => current_user_can('administrator')
            ],
            'plugin_url' => $this->plugin_url
        ]);
        
        // Load WordPress components
        wp_enqueue_script('wp-components');
        wp_enqueue_style('wp-components');
    }

    public function render_admin_page() {
        // Security check
        if (!current_user_can($this->capability)) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'arix-pro-importer'));
        }
        
        // Get current step from query param or session
        $step = isset($_GET['step']) ? sanitize_key($_GET['step']) : 'upload';
        $steps = ['upload', 'mapping', 'settings', 'importing', 'complete'];
        $current_step = in_array($step, $steps) ? $step : 'upload';
        
        // Load template based on step
        include $this->plugin_path . 'templates/admin-page.php';
    }

    // =============== CSV HANDLING ===============
    public function handle_csv_upload() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        if (empty($_FILES['csv_file'])) {
            wp_send_json_error(['message' => __('No file uploaded', 'arix-pro-importer')]);
        }
        
        // Validate file type
        $file_type = wp_check_filetype($_FILES['csv_file']['name']);
        if (!in_array($file_type['ext'], ['csv', 'txt'])) {
            wp_send_json_error(['message' => __('Invalid file type. Please upload a CSV file.', 'arix-pro-importer')]);
        }
        
        // Handle large files - store in uploads directory
        $upload_dir = wp_upload_dir();
        $import_dir = $upload_dir['basedir'] . '/arix-importer/';
        
        // Create directory if not exists
        if (!file_exists($import_dir)) {
            wp_mkdir_p($import_dir);
        }
        
        // Generate unique filename
        $filename = 'import-' . current_time('YmdHis') . '-' . uniqid() . '.csv';
        $filepath = $import_dir . $filename;
        
        // Move uploaded file
        if (move_uploaded_file($_FILES['csv_file']['tmp_name'], $filepath)) {
            // Store in session
            $this->import_session['csv_file'] = $filepath;
            $this->import_session['csv_filename'] = $_FILES['csv_file']['name'];
            $this->save_import_session();
            
            wp_send_json_success([
                'file_id' => $filename,
                'filepath' => $filepath,
                'message' => __('File uploaded successfully', 'arix-pro-importer')
            ]);
        } else {
            wp_send_json_error(['message' => __('File upload failed', 'arix-pro-importer')]);
        }
    }

    public function parse_csv_preview() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $filepath = isset($_POST['filepath']) ? sanitize_text_field(wp_unslash($_POST['filepath'])) : '';
        
        if (!$filepath || !file_exists($filepath)) {
            wp_send_json_error(['message' => __('CSV file not found', 'arix-pro-importer')]);
        }
        
        // Initialize CSV parser
        require_once $this->plugin_path . 'includes/class-csv-parser.php';
        $parser = new ARIX_CSV_Parser($filepath);
        
        // Get preview data (first 5 rows + headers)
        $preview = $parser->get_preview(5);
        
        if (is_wp_error($preview)) {
            wp_send_json_error(['message' => $preview->get_error_message()]);
        }
        
        // Detect columns and sample data
        wp_send_json_success([
            'headers' => $preview['headers'],
            'sample_rows' => $preview['rows'],
            'total_rows' => $preview['total_rows'],
            'detected_delimiter' => $preview['delimiter'],
            'encoding' => $preview['encoding'],
            'has_bom' => $preview['has_bom']
        ]);
    }

    // =============== MAPPING TEMPLATES ===============
    public function save_mapping_template() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $template_name = isset($_POST['template_name']) ? sanitize_text_field(wp_unslash($_POST['template_name'])) : '';
        $mapping_data = isset($_POST['mapping']) ? wp_unslash($_POST['mapping']) : [];
        
        if (empty($template_name) || empty($mapping_data)) {
            wp_send_json_error(['message' => __('Template name and mapping data are required', 'arix-pro-importer')]);
        }
        
        // Get existing templates
        $templates = get_option('arix_mapping_templates', []);
        
        // Save new template
        $template_id = 'tmpl_' . md5($template_name . time());
        $templates[$template_id] = [
            'id' => $template_id,
            'name' => $template_name,
            'mapping' => $mapping_data,
            'created' => current_time('mysql'),
            'columns' => isset($_POST['columns']) ? $_POST['columns'] : []
        ];
        
        update_option('arix_mapping_templates', $templates);
        
        wp_send_json_success([
            'template_id' => $template_id,
            'message' => __('Mapping template saved successfully!', 'arix-pro-importer')
        ]);
    }

    public function load_mapping_template() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $template_id = isset($_POST['template_id']) ? sanitize_text_field(wp_unslash($_POST['template_id'])) : '';
        
        if (empty($template_id)) {
            wp_send_json_error(['message' => __('Invalid template ID', 'arix-pro-importer')]);
        }
        
        $templates = get_option('arix_mapping_templates', []);
        
        if (!isset($templates[$template_id])) {
            wp_send_json_error(['message' => __('Template not found', 'arix-pro-importer')]);
        }
        
        wp_send_json_success([
            'template' => $templates[$template_id],
            'message' => __('Template loaded successfully', 'arix-pro-importer')
        ]);
    }

    public function delete_mapping_template() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $template_id = isset($_POST['template_id']) ? sanitize_text_field(wp_unslash($_POST['template_id'])) : '';
        
        if (empty($template_id)) {
            wp_send_json_error(['message' => __('Invalid template ID', 'arix-pro-importer')]);
        }
        
        $templates = get_option('arix_mapping_templates', []);
        
        if (isset($templates[$template_id])) {
            unset($templates[$template_id]);
            update_option('arix_mapping_templates', $templates);
            wp_send_json_success(['message' => __('Template deleted successfully', 'arix-pro-importer')]);
        } else {
            wp_send_json_error(['message' => __('Template not found', 'arix-pro-importer')]);
        }
    }

    // =============== IMPORT ENGINE ===============
    public function start_import_process() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        // Validate required data
        $required = ['csv_file', 'mapping', 'import_settings'];
        foreach ($required as $field) {
            if (empty($_POST[$field])) {
                wp_send_json_error(['message' => sprintf(__('Missing required field: %s', 'arix-pro-importer'), $field)]);
            }
        }
        
        // Initialize import session
        $this->import_session = [
            'csv_file' => sanitize_text_field(wp_unslash($_POST['csv_file'])),
            'mapping' => wp_unslash($_POST['mapping']),
            'settings' => wp_unslash($_POST['import_settings']),
            'current_row' => 1, // Start after header row
            'total_rows' => intval($_POST['total_rows']),
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => [],
            'start_time' => time(),
            'status' => 'processing',
            'import_id' => 'imp_' . uniqid()
        ];
        
        // Save session
        $this->save_import_session();
        
        // Prepare response
        wp_send_json_success([
            'import_id' => $this->import_session['import_id'],
            'total_rows' => $this->import_session['total_rows'] - 1, // Exclude header
            'message' => __('Import process initialized successfully', 'arix-pro-importer')
        ]);
    }

    public function process_import_chunk() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $import_id = isset($_POST['import_id']) ? sanitize_text_field(wp_unslash($_POST['import_id'])) : '';
        
        // Load import session
        if (!$this->load_import_session($import_id)) {
            wp_send_json_error(['message' => __('Import session not found', 'arix-pro-importer')]);
        }
        
        // Validate session state
        if ($this->import_session['status'] !== 'processing') {
            wp_send_json_error(['message' => __('Import session is not active', 'arix-pro-importer')]);
        }
        
        // Process chunk
        require_once $this->plugin_path . 'includes/class-product-importer.php';
        $importer = new ARIX_Product_Importer($this->import_session);
        $result = $importer->process_chunk();
        
        // Update session
        $this->import_session = $importer->get_session();
        $this->save_import_session();
        
        // Check if import is complete
        $is_complete = ($this->import_session['current_row'] >= $this->import_session['total_rows']);
        
        wp_send_json_success([
            'processed' => $this->import_session['processed'],
            'success' => $this->import_session['success'],
            'failed' => $this->import_session['failed'],
            'current_row' => $this->import_session['current_row'],
            'total_rows' => $this->import_session['total_rows'] - 1,
            'errors' => array_slice($this->import_session['errors'], -5), // Last 5 errors
            'is_complete' => $is_complete,
            'progress' => round(($this->import_session['current_row'] / $this->import_session['total_rows']) * 100, 1),
            'elapsed' => time() - $this->import_session['start_time']
        ]);
    }

    public function resume_import() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $import_id = isset($_POST['import_id']) ? sanitize_text_field(wp_unslash($_POST['import_id'])) : '';
        
        if (!$this->load_import_session($import_id)) {
            wp_send_json_error(['message' => __('Import session not found', 'arix-pro-importer')]);
        }
        
        // Resume processing
        $this->import_session['status'] = 'processing';
        $this->save_import_session();
        
        wp_send_json_success([
            'message' => __('Import resumed successfully', 'arix-pro-importer'),
            'current_row' => $this->import_session['current_row'],
            'total_rows' => $this->import_session['total_rows'] - 1
        ]);
    }

    public function cancel_import() {
        check_ajax_referer('arix_importer_nonce', 'nonce');
        
        if (!current_user_can($this->capability)) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'arix-pro-importer')]);
        }
        
        $import_id = isset($_POST['import_id']) ? sanitize_text_field(wp_unslash($_POST['import_id'])) : '';
        
        if ($this->load_import_session($import_id)) {
            // Clean up session
            $this->delete_import_session($import_id);
            
            // Clean up CSV file if exists
            if (!empty($this->import_session['csv_file']) && file_exists($this->import_session['csv_file'])) {
                unlink($this->import_session['csv_file']);
            }
        }
        
        wp_send_json_success(['message' => __('Import cancelled and cleaned up', 'arix-pro-importer')]);
    }

    // =============== DRAFT ENFORCEMENT ===============
    public function enforce_draft_status($data, $postarr) {
        // Only enforce for products being imported
        if (defined('ARIX_IMPORT_PROCESS') && ARIX_IMPORT_PROCESS && 
            isset($postarr['post_type']) && $postarr['post_type'] === 'product') {
            $data['post_status'] = 'draft';
        }
        return $data;
    }

    public function enforce_draft_on_variations($product_data, $variation_data) {
        if (defined('ARIX_IMPORT_PROCESS') && ARIX_IMPORT_PROCESS) {
            $product_data['post_status'] = 'draft';
        }
        return $product_data;
    }

    // =============== SESSION MANAGEMENT ===============
    public function init_import_session() {
        if (!session_id()) {
            session_start();
        }
    }

    public function save_import_session() {
        if (!empty($this->import_session['import_id'])) {
            set_transient('arix_import_' . $this->import_session['import_id'], $this->import_session, HOUR_IN_SECONDS * 2);
        }
    }

    public function load_import_session($import_id) {
        $session = get_transient('arix_import_' . $import_id);
        if ($session) {
            $this->import_session = $session;
            return true;
        }
        return false;
    }

    public function delete_import_session($import_id) {
        delete_transient('arix_import_' . $import_id);
        $this->import_session = [];
    }
}

// Initialize the plugin
add_action('plugins_loaded', function() {
    ARIX_Pro_Importer::instance();
});

// Autoloader for includes
spl_autoload_register(function ($class) {
    if (strpos($class, 'ARIX_') === 0) {
        $class_file = plugin_dir_path(__FILE__) . 'includes/class-' . strtolower(str_replace('_', '-', $class)) . '.php';
        if (file_exists($class_file)) {
            require_once $class_file;
        }
    }
});