<?php
class ARIX_CSV_Parser {
    private $filepath;
    private $encoding;
    private $delimiter;
    private $has_bom = false;
    
    public function __construct($filepath) {
        $this->filepath = $filepath;
        $this->detect_encoding();
        $this->detect_delimiter();
    }
    
    private function detect_encoding() {
        $sample = file_get_contents($this->filepath, false, null, 0, 1000);
        
        // Check for BOM
        if (substr($sample, 0, 3) === "\xEF\xBB\xBF") {
            $this->encoding = 'UTF-8-BOM';
            $this->has_bom = true;
        } elseif (mb_detect_encoding($sample, 'UTF-8, ISO-8859-1, Windows-1252', true)) {
            $this->encoding = mb_detect_encoding($sample, 'UTF-8, ISO-8859-1, Windows-1252', true);
        } else {
            $this->encoding = 'UTF-8';
        }
    }
    
    private function detect_delimiter() {
        $file = fopen($this->filepath, 'r');
        if (!$file) return;
        
        // Read first line
        $sample = fgets($file);
        fclose($file);
        
        // Try common delimiters
        $delimiters = [',', ';', "\t", '|', ':'];
        $delimiter_counts = [];
        
        foreach ($delimiters as $delimiter) {
            $delimiter_counts[$delimiter] = substr_count($sample, $delimiter);
        }
        
        // Get delimiter with highest count
        arsort($delimiter_counts);
        $this->delimiter = key($delimiter_counts);
    }
    
    public function get_preview($row_count = 5) {
        $rows = [];
        $headers = [];
        $total_rows = 0;
        
        if (!file_exists($this->filepath)) {
            return new WP_Error('file_not_found', __('CSV file not found', 'arix-pro-importer'));
        }
        
        $file = fopen($this->filepath, 'r');
        if (!$file) {
            return new WP_Error('file_open_failed', __('Could not open CSV file', 'arix-pro-importer'));
        }
        
        // Handle BOM
        if ($this->has_bom) {
            fgets($file); // Skip BOM
        }
        
        // Read headers
        $headers_line = fgets($file);
        if ($this->encoding !== 'UTF-8') {
            $headers_line = mb_convert_encoding($headers_line, 'UTF-8', $this->encoding);
        }
        $headers = str_getcsv($headers_line, $this->delimiter);
        $headers = array_map('trim', $headers);
        
        // Read sample rows
        $row_num = 0;
        while (($row = fgetcsv($file, 0, $this->delimiter)) !== false) {
            $total_rows++;
            
            if ($row_num < $row_count) {
                // Convert encoding if needed
                if ($this->encoding !== 'UTF-8') {
                    $row = array_map(function($item) {
                        return mb_convert_encoding($item, 'UTF-8', $this->encoding);
                    }, $row);
                }
                
                // Pad row to match header count
                if (count($row) < count($headers)) {
                    $row = array_pad($row, count($headers), '');
                }
                
                $rows[] = array_combine($headers, $row);
                $row_num++;
            }
        }
        
        fclose($file);
        
        return [
            'headers' => $headers,
            'rows' => $rows,
            'total_rows' => $total_rows + 1, // +1 for header row
            'delimiter' => $this->delimiter,
            'encoding' => $this->encoding,
            'has_bom' => $this->has_bom
        ];
    }
    
    public function get_chunk($start_row, $chunk_size) {
        $rows = [];
        $file = fopen($this->filepath, 'r');
        if (!$file) return $rows;
        
        // Skip BOM if present
        if ($this->has_bom) {
            fgets($file);
        }
        
        // Skip to start row
        for ($i = 0; $i < $start_row; $i++) {
            if (feof($file)) break;
            fgets($file);
        }
        
        // Read chunk
        $row_count = 0;
        while (($row = fgetcsv($file, 0, $this->delimiter)) !== false && $row_count < $chunk_size) {
            if ($this->encoding !== 'UTF-8') {
                $row = array_map(function($item) {
                    return mb_convert_encoding($item, 'UTF-8', $this->encoding);
                }, $row);
            }
            $rows[] = $row;
            $row_count++;
        }
        
        fclose($file);
        return $rows;
    }
}