(function($) {
    'use strict';
    
    window.ARIXImporter = window.ARIXImporter || {};
    
    $(document).ready(function() {
        ARIXImporter.init();
    });
    
    window.ARIXImporter = {
        currentStep: 'upload',
        importSession: null,
        csvData: null,
        mapping: {},
        settings: {
            chunk_size: arixImporter.i18n.chunk_size,
            skip_errors: true,
            dry_run: false
        },
        
        init: function() {
            this.cacheElements();
            this.bindEvents();
            this.loadSavedTemplates();
            this.renderStep(this.currentStep);
        },
        
        cacheElements: function() {
            this.$container = $('#arix-importer-app');
            this.$steps = $('.arix-wizard-step');
            this.$sidebarSteps = $('.arix-step-item');
            this.$dropzone = $('.arix-dropzone');
            this.$fileInput = $('#csv_file_input');
            this.$previewTable = $('#csv_preview_table');
            this.$mappingContainer = $('#mapping_container');
            this.$templateSelect = $('#template_select');
            this.$progressBar = $('.arix-progress-fill');
            this.$importSummary = $('#import_summary');
        },
        
        bindEvents: function() {
            // File upload handling
            this.$dropzone.on('click', () => this.$fileInput.click());
            this.$fileInput.on('change', (e) => this.handleFileSelect(e));
            this.$dropzone.on('dragover', (e) => this.handleDragOver(e));
            this.$dropzone.on('dragleave', (e) => this.handleDragLeave(e));
            this.$dropzone.on('drop', (e) => this.handleDrop(e));
            
            // Navigation
            $('.arix-step-link').on('click', (e) => this.navigateStep(e));
            $('#btn_next_upload').on('click', () => this.nextStep());
            $('#btn_next_mapping').on('click', () => this.saveMappingAndNext());
            $('#btn_start_import').on('click', () => this.startImport());
            $('#btn_resume_import').on('click', () => this.resumeImport());
            $('#btn_cancel_import').on('click', () => this.cancelImport());
            
            // Mapping interactions
            $('#save_template_btn').on('click', () => this.saveMappingTemplate());
            $('#load_template_btn').on('click', () => this.loadSelectedTemplate());
            $('#delete_template_btn').on('click', () => this.deleteSelectedTemplate());
            
            // Settings
            $('#dry_run_toggle').on('change', (e) => {
                this.settings.dry_run = e.target.checked;
                this.updateDraftWarning();
            });
            
            // Make columns draggable
            $(document).on('mousedown', '.arix-column-item', (e) => this.startDrag(e));
        },
        
        handleFileSelect: function(e) {
            const file = e.target.files[0];
            if (!file) return;
            
            this.validateAndUploadFile(file);
        },
        
        validateAndUploadFile: function(file) {
            // Validate file size
            if (file.size > arixImporter.i18n.max_file_size) {
                this.showError(arixImporter.i18n.file_too_large);
                return;
            }
            
            // Validate file type
            if (!file.name.match(/\.(csv|txt)$/i)) {
                this.showError(arixImporter.i18n.invalid_file_type);
                return;
            }
            
            // Show loading state
            this.showLoading(this.$dropzone, arixImporter.i18n.uploading);
            
            // Upload file via AJAX
            const formData = new FormData();
            formData.append('action', 'arix_upload_csv');
            formData.append('nonce', arixImporter.nonce);
            formData.append('csv_file', file);
            
            $.ajax({
                url: arixImporter.ajax_url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: (response) => {
                    if (response.success) {
                        this.parseCSVPreview(response.data.filepath);
                    } else {
                        this.showError(response.data.message || arixImporter.i18n.upload_failed);
                    }
                },
                error: () => {
                    this.showError(arixImporter.i18n.upload_failed);
                },
                complete: () => {
                    this.hideLoading(this.$dropzone);
                }
            });
        },
        
        parseCSVPreview: function(filepath) {
            this.showLoading(this.$container, arixImporter.i18n.processing);
            
            $.post(arixImporter.ajax_url, {
                action: 'arix_parse_csv_preview',
                nonce: arixImporter.nonce,
                filepath: filepath
            }, (response) => {
                if (response.success) {
                    this.csvData = response.data;
                    this.renderCSVPreview();
                    this.enableNextStep('upload');
                } else {
                    this.showError(response.data.message || arixImporter.i18n.invalid_csv);
                }
            }).fail(() => {
                this.showError(arixImporter.i18n.invalid_csv);
            }).always(() => {
                this.hideLoading(this.$container);
            });
        },
        
        renderCSVPreview: function() {
            if (!this.csvData || !this.csvData.headers) return;
            
            let tableHTML = `
                <div class="arix-section-title">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"></path>
                    </svg>
                    CSV Preview (${this.csvData.total_rows} rows)
                </div>
                <div class="arix-table-container" style="overflow-x: auto; border-radius: 12px; box-shadow: var(--arix-shadow);">
                    <table class="wp-list-table widefat fixed striped" style="min-width: 100%; border-radius: 12px; overflow: hidden;">
                        <thead>
                            <tr>
                                ${this.csvData.headers.map(header => `<th>${this.escapeHtml(header)}</th>`).join('')}
                            </tr>
                        </thead>
                        <tbody>
            `;
            
            this.csvData.sample_rows.forEach(row => {
                tableHTML += '<tr>';
                this.csvData.headers.forEach(header => {
                    const value = row[header] || '';
                    tableHTML += `<td>${this.escapeHtml(value.substring(0, 50))}${value.length > 50 ? '...' : ''}</td>`;
                });
                tableHTML += '</tr>';
            });
            
            tableHTML += `
                        </tbody>
                    </table>
                </div>
                <div class="arix-file-info" style="margin-top: 1.5rem; padding: 1rem; background: var(--arix-sidebar-bg); border-radius: 8px;">
                    <p><strong>Detected Settings:</strong></p>
                    <ul style="margin: 0.5rem 0 0 1.5rem; color: var(--arix-text-secondary);">
                        <li>Delimiter: <code>${this.csvData.detected_delimiter}</code></li>
                        <li>Encoding: <code>${this.csvData.encoding}${this.csvData.has_bom ? ' (with BOM)' : ''}</code></li>
                        <li>Total Rows: <strong>${this.csvData.total_rows.toLocaleString()}</strong></li>
                    </ul>
                </div>
            `;
            
            $('#csv_preview_container').html(tableHTML);
        },
        
        renderMappingUI: function() {
            if (!this.csvData || !this.csvData.headers) return;
            
            // CSV Columns Panel
            let csvColumnsHTML = `
                <div class="arix-section-title">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                    CSV Columns
                </div>
                <p style="margin-bottom: 1rem; color: var(--arix-text-secondary);">Drag columns to the right panel to map them to WooCommerce fields</p>
                <div id="csv_columns_list">
            `;
            
            this.csvData.headers.forEach(header => {
                csvColumnsHTML += `
                    <div class="arix-column-item" data-column="${this.escapeHtml(header)}">
                        ${this.escapeHtml(header)}
                    </div>
                `;
            });
            
            csvColumnsHTML += `</div>`;
            
            // Mapped Fields Panel
            const woocommerceFields = [
                { id: 'name', label: 'Product Name', required: true, description: 'Main product title' },
                { id: 'sku', label: 'SKU', required: false, description: 'Stock keeping unit (must be unique)' },
                { id: 'type', label: 'Product Type', required: false, description: 'simple, variable, variation' },
                { id: 'parent_sku', label: 'Parent SKU', required: false, description: 'For variations only' },
                { id: 'regular_price', label: 'Regular Price', required: false, description: 'Numeric value' },
                { id: 'sale_price', label: 'Sale Price', required: false, description: 'Numeric value' },
                { id: 'stock_quantity', label: 'Stock Quantity', required: false, description: 'Number of items in stock' },
                { id: 'stock_status', label: 'Stock Status', required: false, description: 'instock, outofstock, onbackorder' },
                { id: 'categories', label: 'Categories', required: false, description: 'Comma separated category names' },
                { id: 'tags', label: 'Tags', required: false, description: 'Comma separated tag names' },
                { id: 'featured_image', label: 'Featured Image', required: false, description: 'URL to main product image' },
                { id: 'gallery_images', label: 'Gallery Images', required: false, description: 'Comma separated URLs' },
                { id: 'description', label: 'Description', required: false, description: 'Full product description' },
                { id: 'short_description', label: 'Short Description', required: false, description: 'Excerpt for catalog' },
                { id: 'weight', label: 'Weight', required: false, description: 'Product weight (decimal)' },
                { id: 'length', label: 'Length', required: false, description: 'Product length (decimal)' },
                { id: 'width', label: 'Width', required: false, description: 'Product width (decimal)' },
                { id: 'height', label: 'Height', required: false, description: 'Product height (decimal)' },
                { id: 'attributes', label: 'Attributes', required: false, description: 'Special mapping interface' },
                { id: 'meta_data', label: 'Custom Meta', required: false, description: 'Map to custom fields' }
            ];
            
            let mappedFieldsHTML = `
                <div class="arix-section-title">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M4 6h16M4 12h16M4 18h16"></path>
                    </svg>
                    WooCommerce Fields
                </div>
                <p style="margin-bottom: 1rem; color: var(--arix-text-secondary);">Drop CSV columns here to map them to product fields</p>
                <div id="mapped_fields_list">
            `;
            
            woocommerceFields.forEach(field => {
                const isMapped = this.mapping[field.id] ? 'has-mapping' : '';
                const mappedValue = this.mapping[field.id] ? `<div class="arix-mapped-field-value">${this.escapeHtml(this.mapping[field.id])}</div>` : '';
                
                mappedFieldsHTML += `
                    <div class="arix-mapped-field ${isMapped}" data-field="${field.id}">
                        <label class="arix-mapped-field-label">
                            ${field.label} ${field.required ? '<span style="color: var(--arix-danger);">*</span>' : ''}
                        </label>
                        <div class="arix-mapped-field-description">${field.description}</div>
                        ${mappedValue}
                    </div>
                `;
            });
            
            mappedFieldsHTML += `</div>`;
            
            // Template Management
            const templateHTML = `
                <div class="arix-section-title" style="margin-top: 2rem;">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M5 7h14M5 12h14M5 17h14"></path>
                    </svg>
                    Mapping Templates
                </div>
                <div class="arix-template-actions">
                    <div style="flex: 1;">
                        <select id="template_select" class="components-select-control__input" style="width: 100%; padding: 0.75rem; border-radius: 8px; border: 1px solid var(--arix-border);">
                            <option value="">-- Select a template --</option>
                        </select>
                    </div>
                    <button id="load_template_btn" class="arix-btn arix-btn-secondary" style="padding: 0.75rem 1.5rem;">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 5px;">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"></path>
                        </svg>
                        Load Template
                    </button>
                    <button id="save_template_btn" class="arix-btn" style="padding: 0.75rem 1.5rem;">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 5px;">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M8 7H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-3m-4 0V5a2 2 0 114 0v2M9 7h6"></path>
                        </svg>
                        Save as Template
                    </button>
                    <button id="delete_template_btn" class="arix-btn arix-btn-secondary" style="padding: 0.75rem 1.5rem; background: var(--arix-danger); color: white; border-color: var(--arix-danger);">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 5px;">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                        </svg>
                        Delete
                    </button>
                </div>
                <div class="arix-draft-badge" style="margin-top: 1.5rem;">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <line x1="12" y1="8" x2="12" y2="12"></line>
                        <line x1="12" y1="16" x2="12.01" y2="16"></line>
                    </svg>
                    ${arixImporter.i18n.draft_warning}
                </div>
            `;
            
            this.$mappingContainer.html(`
                <div class="arix-mapping-container">
                    <div class="arix-csv-columns">${csvColumnsHTML}</div>
                    <div class="arix-mapped-fields">${mappedFieldsHTML}</div>
                </div>
                ${templateHTML}
            `);
            
            // Make fields droppable
            $('.arix-mapped-field').on('dragover', (e) => {
                e.preventDefault();
                $(e.currentTarget).css('border-color', 'var(--arix-primary)');
            });
            
            $('.arix-mapped-field').on('dragleave', (e) => {
                $(e.currentTarget).css('border-color', '');
            });
            
            $('.arix-mapped-field').on('drop', (e) => {
                e.preventDefault();
                const $field = $(e.currentTarget);
                const fieldId = $field.data('field');
                const column = this.draggedColumn;
                
                if (column) {
                    this.mapping[fieldId] = column;
                    $field.addClass('has-mapping').html(`
                        <label class="arix-mapped-field-label">
                            ${$field.find('.arix-mapped-field-label').text()}
                        </label>
                        <div class="arix-mapped-field-description">${$field.find('.arix-mapped-field-description').text()}</div>
                        <div class="arix-mapped-field-value">${this.escapeHtml(column)}</div>
                    `);
                    this.draggedColumn = null;
                }
                
                $field.css('border-color', '');
            });
        },
        
        // Additional methods for drag/drop, template management, import process, etc.
        // (Full implementation would include all required functionality)
        
        startDrag: function(e) {
            this.draggedColumn = $(e.currentTarget).data('column');
            $(e.currentTarget).addClass('dragging');
            
            $(document).on('mouseup', () => {
                $('.arix-column-item').removeClass('dragging');
                $(document).off('mouseup');
            });
        },
        
        loadSavedTemplates: function() {
            // In a real implementation, this would fetch templates via AJAX
            // For demo purposes, we'll simulate with localStorage
            const templates = JSON.parse(localStorage.getItem('arix_mapping_templates') || '[]');
            
            if (templates.length > 0) {
                let options = '<option value="">-- Select a template --</option>';
                templates.forEach(template => {
                    options += `<option value="${template.id}">${this.escapeHtml(template.name)}</option>`;
                });
                this.$templateSelect.html(options);
            } else {
                this.$templateSelect.html('<option value="">-- No saved templates --</option>');
            }
        },
        
        saveMappingTemplate: function() {
            const templateName = prompt('Enter template name:');
            if (!templateName) return;
            
            const template = {
                id: 'tmpl_' + Date.now(),
                name: templateName,
                mapping: this.mapping,
                columns: this.csvData.headers,
                created: new Date().toISOString()
            };
            
            // Save to localStorage (in real plugin, this would be AJAX to server)
            let templates = JSON.parse(localStorage.getItem('arix_mapping_templates') || '[]');
            templates.push(template);
            localStorage.setItem('arix_mapping_templates', JSON.stringify(templates));
            
            this.loadSavedTemplates();
            this.showNotice(arixImporter.i18n.template_saved, 'success');
        },
        
        loadSelectedTemplate: function() {
            const templateId = this.$templateSelect.val();
            if (!templateId) return;
            
            const templates = JSON.parse(localStorage.getItem('arix_mapping_templates') || '[]');
            const template = templates.find(t => t.id === templateId);
            
            if (template) {
                // Verify column compatibility
                const missingColumns = template.columns.filter(col => !this.csvData.headers.includes(col));
                
                if (missingColumns.length > 0) {
                    if (!confirm(`The selected template was created with different columns. Missing columns: ${missingColumns.join(', ')}. Continue anyway?`)) {
                        return;
                    }
                }
                
                this.mapping = template.mapping;
                this.renderMappingUI();
                this.showNotice('Template loaded successfully!', 'success');
            }
        },
        
        deleteSelectedTemplate: function() {
            const templateId = this.$templateSelect.val();
            if (!templateId) return;
            
            if (!confirm('Are you sure you want to delete this template?')) return;
            
            let templates = JSON.parse(localStorage.getItem('arix_mapping_templates') || '[]');
            templates = templates.filter(t => t.id !== templateId);
            localStorage.setItem('arix_mapping_templates', JSON.stringify(templates));
            
            this.loadSavedTemplates();
            this.showNotice(arixImporter.i18n.template_deleted, 'success');
        },
        
        startImport: function() {
            if (Object.keys(this.mapping).length === 0) {
                this.showError('Please map at least one column before importing');
                return;
            }
            
            // Show loading state
            this.showLoading(this.$container, 'Initializing import...');
            
            // Prepare import data
            const importData = {
                action: 'arix_start_import',
                nonce: arixImporter.nonce,
                csv_file: this.csvData.filepath,
                mapping: this.mapping,
                import_settings: this.settings,
                total_rows: this.csvData.total_rows
            };
            
            // Start import process
            $.post(arixImporter.ajax_url, importData, (response) => {
                if (response.success) {
                    this.importSession = response.data;
                    this.currentStep = 'importing';
                    this.renderStep('importing');
                    this.processImportChunk();
                } else {
                    this.showError(response.data.message || 'Failed to start import');
                }
            }).fail(() => {
                this.showError('Import initialization failed');
            }).always(() => {
                this.hideLoading(this.$container);
            });
        },
        
        processImportChunk: function() {
            if (!this.importSession) return;
            
            $.post(arixImporter.ajax_url, {
                action: 'arix_process_chunk',
                nonce: arixImporter.nonce,
                import_id: this.importSession.import_id
            }, (response) => {
                if (response.success) {
                    // Update progress
                    const progress = response.data.progress || 0;
                    this.$progressBar.css('width', progress + '%');
                    
                    // Update stats
                    $('#processed_count').text(response.data.processed.toLocaleString());
                    $('#success_count').text(response.data.success.toLocaleString());
                    $('#failed_count').text(response.data.failed.toLocaleString());
                    $('#elapsed_time').text(this.formatTime(response.data.elapsed));
                    
                    // Show errors if any
                    if (response.data.errors && response.data.errors.length > 0) {
                        let errorHTML = '<div style="margin-top: 1.5rem; padding: 1rem; background: rgba(239, 68, 68, 0.1); border-radius: 8px; border-left: 3px solid var(--arix-danger);">';
                        errorHTML += '<strong>Recent Errors:</strong><ul style="margin: 0.5rem 0 0 1.5rem;">';
                        
                        response.data.errors.forEach(error => {
                            errorHTML += `<li>Row ${error.row}: ${this.escapeHtml(error.message)}</li>`;
                        });
                        
                        errorHTML += '</ul></div>';
                        $('#import_errors').html(errorHTML);
                    }
                    
                    // Check if import is complete
                    if (response.data.is_complete) {
                        this.finalizeImport();
                    } else {
                        // Process next chunk after delay
                        setTimeout(() => this.processImportChunk(), 300);
                    }
                } else {
                    this.showError(response.data.message || 'Import chunk failed');
                }
            }).fail(() => {
                this.showError('Import chunk failed. You can resume from where it left off.');
            });
        },
        
        finalizeImport: function() {
            // Show completion summary
            this.currentStep = 'complete';
            this.renderStep('complete');
            
            // Update summary stats
            $('#total_rows_summary').text((this.importSession.total_rows - 1).toLocaleString());
            $('#success_rows_summary').text(this.importSession.success.toLocaleString());
            $('#failed_rows_summary').text(this.importSession.failed.toLocaleString());
            $('#elapsed_time_summary').text(this.formatTime(Date.now() - this.importSession.start_time * 1000));
            
            // Show success message
            this.showNotice(`Import completed! ${this.importSession.success} products created as drafts.`, 'success');
        },
        
        resumeImport: function() {
            if (!this.importSession) {
                this.showError('No import session found to resume');
                return;
            }
            
            $.post(arixImporter.ajax_url, {
                action: 'arix_resume_import',
                nonce: arixImporter.nonce,
                import_id: this.importSession.import_id
            }, (response) => {
                if (response.success) {
                    this.showNotice(arixImporter.i18n.import_resumed, 'success');
                    this.processImportChunk();
                } else {
                    this.showError(response.data.message || 'Failed to resume import');
                }
            });
        },
        
        cancelImport: function() {
            if (!this.importSession) return;
            
            if (!confirm('Are you sure you want to cancel this import? All progress will be lost.')) return;
            
            $.post(arixImporter.ajax_url, {
                action: 'arix_cancel_import',
                nonce: arixImporter.nonce,
                import_id: this.importSession.import_id
            }, (response) => {
                if (response.success) {
                    this.importSession = null;
                    this.currentStep = 'upload';
                    this.renderStep('upload');
                    this.showNotice(arixImporter.i18n.import_cancelled, 'success');
                } else {
                    this.showError(response.data.message || 'Failed to cancel import');
                }
            });
        },
        
        // Utility methods
        navigateStep: function(e) {
            e.preventDefault();
            const targetStep = $(e.currentTarget).closest('.arix-step-item').data('step');
            if (targetStep && this.validateStep(targetStep)) {
                this.currentStep = targetStep;
                this.renderStep(targetStep);
            }
        },
        
        nextStep: function() {
            const steps = ['upload', 'mapping', 'settings', 'importing', 'complete'];
            const currentIndex = steps.indexOf(this.currentStep);
            if (currentIndex < steps.length - 1 && this.validateStep(steps[currentIndex + 1])) {
                this.currentStep = steps[currentIndex + 1];
                this.renderStep(this.currentStep);
            }
        },
        
        validateStep: function(step) {
            if (step === 'mapping' && !this.csvData) {
                this.showError('Please upload a CSV file first');
                return false;
            }
            if (step === 'importing' && Object.keys(this.mapping).length === 0) {
                this.showError('Please map at least one column before importing');
                return false;
            }
            return true;
        },
        
        renderStep: function(step) {
            // Update sidebar active step
            this.$sidebarSteps.removeClass('active completed');
            this.$sidebarSteps.each(function() {
                const stepId = $(this).data('step');
                if (stepId === step) {
                    $(this).addClass('active');
                } else if (['upload', 'mapping', 'settings'].includes(stepId) && 
                          ['mapping', 'settings', 'importing', 'complete'].includes(step)) {
                    $(this).addClass('completed');
                }
            });
            
            // Show current step content
            this.$steps.removeClass('active');
            $(`#step_${step}`).addClass('active');
            
            // Special rendering for each step
            if (step === 'mapping' && !this.mappingRendered) {
                this.renderMappingUI();
                this.mappingRendered = true;
            }
            
            if (step === 'settings') {
                this.renderSettingsStep();
            }
            
            // Scroll to top
            window.scrollTo(0, 0);
        },
        
        renderSettingsStep: function() {
            $('#import_settings_container').html(`
                <div class="arix-section-title">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"></path>
                        <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                    </svg>
                    Import Settings
                </div>
                <div style="background: var(--arix-sidebar-bg); border-radius: 12px; padding: 1.5rem; margin: 1.5rem 0;">
                    <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 1.25rem;">
                        <div>
                            <h3 style="margin: 0 0 0.25rem 0; font-weight: 600;">Dry Run Mode</h3>
                            <p style="margin: 0; color: var(--arix-text-secondary);">Simulate import without creating any products</p>
                        </div>
                        <label class="components-form-toggle" style="margin: 0;">
                            <input type="checkbox" id="dry_run_toggle" ${this.settings.dry_run ? 'checked' : ''}>
                            <span class="components-form-toggle__track"></span>
                            <span class="components-form-toggle__thumb"></span>
                        </label>
                    </div>
                    
                    <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 1.25rem;">
                        <div>
                            <h3 style="margin: 0 0 0.25rem 0; font-weight: 600;">Skip Errors</h3>
                            <p style="margin: 0; color: var(--arix-text-secondary);">Continue importing even if some rows fail</p>
                        </div>
                        <label class="components-form-toggle" style="margin: 0;">
                            <input type="checkbox" id="skip_errors_toggle" checked>
                            <span class="components-form-toggle__track"></span>
                            <span class="components-form-toggle__thumb"></span>
                        </label>
                    </div>
                    
                    <div style="display: flex; align-items: center; justify-content: space-between;">
                        <div>
                            <h3 style="margin: 0 0 0.25rem 0; font-weight: 600;">Chunk Size</h3>
                            <p style="margin: 0; color: var(--arix-text-secondary);">Number of rows to process per request</p>
                        </div>
                        <input type="number" id="chunk_size_input" value="${this.settings.chunk_size}" min="5" max="100" style="width: 80px; padding: 0.5rem; border-radius: 6px; border: 1px solid var(--arix-border);">
                    </div>
                </div>
                
                <div class="arix-draft-badge">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <line x1="12" y1="8" x2="12" y2="12"></line>
                        <line x1="12" y1="16" x2="12.01" y2="16"></line>
                    </svg>
                    ${arixImporter.i18n.draft_warning}
                </div>
            `);
            
            // Update settings from inputs
            $('#chunk_size_input').on('change', (e) => {
                this.settings.chunk_size = parseInt(e.target.value) || 25;
            });
            
            $('#skip_errors_toggle').on('change', (e) => {
                this.settings.skip_errors = e.target.checked;
            });
        },
        
        updateDraftWarning: function() {
            // This would update the draft warning visibility based on settings
        },
        
        formatTime: function(milliseconds) {
            const seconds = Math.floor(milliseconds / 1000);
            const minutes = Math.floor(seconds / 60);
            const hours = Math.floor(minutes / 60);
            
            if (hours > 0) {
                return `${hours}h ${minutes % 60}m ${seconds % 60}s`;
            } else if (minutes > 0) {
                return `${minutes}m ${seconds % 60}s`;
            } else {
                return `${seconds}s`;
            }
        },
        
        escapeHtml: function(text) {
            if (!text) return '';
            return text.toString()
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#039;');
        },
        
        showLoading: function($element, message) {
            $element.append(`
                <div class="arix-loading-overlay" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; background: rgba(255,255,255,0.9); display: flex; flex-direction: column; align-items: center; justify-content: center; z-index: 1000; border-radius: var(--arix-border-radius);">
                    <div class="arix-loader"></div>
                    <p style="margin-top: 1rem; font-weight: 500;">${message}</p>
                </div>
            `);
        },
        
        hideLoading: function($element) {
            $element.find('.arix-loading-overlay').remove();
        },
        
        showError: function(message) {
            this.showNotice(message, 'error');
        },
        
        showNotice: function(message, type = 'info') {
            // In a real implementation, this would use WordPress notice system
            // For our custom UI, we'll create a temporary notice
            const noticeTypes = {
                'success': {bg: 'rgba(16, 185, 129, 0.15)', border: 'var(--arix-success)', icon: '✓'},
                'error': {bg: 'rgba(239, 68, 68, 0.15)', border: 'var(--arix-danger)', icon: '×'},
                'warning': {bg: 'rgba(245, 158, 11, 0.15)', border: 'var(--arix-warning)', icon: '!'},
                'info': {bg: 'rgba(99, 102, 241, 0.15)', border: 'var(--arix-primary)', icon: 'ℹ'}
            };
            
            const typeConfig = noticeTypes[type] || noticeTypes.info;
            
            const noticeHTML = `
                <div class="arix-notice arix-notice-${type}" style="background: ${typeConfig.bg}; border-left: 4px solid ${typeConfig.border}; padding: 1rem 1.5rem; border-radius: 8px; margin: 1rem 0; display: flex; align-items: flex-start; animation: fadeIn 0.3s;">
                    <div style="font-size: 1.5rem; font-weight: bold; margin-right: 0.75rem; color: ${typeConfig.border};">${typeConfig.icon}</div>
                    <div>${message}</div>
                </div>
            `;
            
            // Remove existing notices
            $('.arix-notice').remove();
            
            // Add new notice
            this.$container.prepend(noticeHTML);
            
            // Auto hide after 5 seconds for success messages
            if (type === 'success') {
                setTimeout(() => {
                    $('.arix-notice').fadeOut(300, function() {
                        $(this).remove();
                    });
                }, 5000);
            }
        },
        
        enableNextStep: function(step) {
            $(`#btn_next_${step}`).prop('disabled', false).removeClass('arix-btn-disabled');
        },
        
        handleDragOver: function(e) {
            e.preventDefault();
            this.$dropzone.addClass('dragover');
        },
        
        handleDragLeave: function(e) {
            e.preventDefault();
            this.$dropzone.removeClass('dragover');
        },
        
        handleDrop: function(e) {
            e.preventDefault();
            this.$dropzone.removeClass('dragover');
            
            if (e.originalEvent.dataTransfer.files.length) {
                this.$fileInput.prop('files', e.originalEvent.dataTransfer.files);
                this.handleFileSelect({ target: this.$fileInput[0] });
            }
        }
    };
    
})(jQuery);