<?php
/**
 * CryptoWoo implementation of The WooCommerce API Manager PHP Client Library
 *
 * @category CryptoWoo
 * @package License
 * @subpackage LicenseMenu
 * Author: CryptoWoo AS
 * Author URI: https://cryptowoo.com
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load WC_AM_Client class if it does not exists.
if ( ! class_exists( 'WC_AM_Client_2_8_1' ) ) {
	require_once __DIR__ . '/api-manager-php-library/wc-am-client.php';
}

if ( ! class_exists( 'CW_License_Menu' ) ) {
	/** CryptoWoo implementation of The WooCommerce API Manager PHP Client Library */
	class CW_License_Menu extends WC_AM_Client_2_8_1 {

		/**
		 * CW_License_Menu constructor.
		 *
		 * @param string            $file File path.
		 * @param int|string        $product_id Product identifier.
		 * @param int|double|string $software_version Software version number.
		 */
		public function __construct( $file, $product_id = '', $software_version = '1.0.0' ) {
			$plugin_data = get_plugin_data( $file );

			parent::__construct( $file, $product_id, $software_version, 'plugin', 'https://www.cryptowoo.com', $plugin_data['Name'], $plugin_data['TextDomain'] );

			remove_action( 'admin_menu', array( $this, 'register_menu' ) );
			add_action( 'admin_menu', array( $this, 'register_submenu' ), 600 );

			$this->maybe_setup_default_data();
		}

		/**
		 * Generate default data if new install because register_activation_hook does not work inside a class.
		 */
		public function maybe_setup_default_data() {
			// No instance id means this is a new install instead of an update.
			if ( false === $this->wc_am_instance_id ) {
				// activation() use wp_generate_password() so it needs to be run in an hook.
				add_action( 'admin_init', function () {
					$this->activation();
					$this->wc_am_instance_id = get_option( $this->wc_am_instance_key );

					// Prevent migrate errors when installing cryptowoo or addon for the first time.
					if ( $this->wc_am_instance_id && has_action( 'admin_notices', array(
							$this,
							'migrate_error_notice'
						) ) ) {
						remove_action( 'admin_notices', array( $this, 'migrate_error_notice' ) );
					}
				} );
			}
		}

		/**
		 * Register submenu specific to this product.
		 */
		public function register_submenu() {
			$settings_menu_title = $this->wc_am_settings_menu_title === 'CryptoWoo Activation' ? $this->wc_am_settings_menu_title : str_replace('CryptoWoo', '', $this->wc_am_settings_menu_title);
			add_submenu_page( 'cryptowoo', $settings_menu_title, $settings_menu_title, 'manage_options', $this->wc_am_activation_tab_key, array(
				$this,
				'config_page',
			) );
		}

		/**
		 * Draw options page and show settings errors.
		 */
		public function config_page() {
			parent::config_page();
			settings_errors();
		}

		/**
		 * Register settings
		 * Overrides parent to rename "Product ID" field label to "License Type"
		 */
		public function load_settings() {
			register_setting( $this->data_key, $this->data_key, array( $this, 'validate_options' ) );
			// API Key
			add_settings_section( $this->wc_am_api_key_key, esc_html__( 'API Key Activation', $this->text_domain ), array(
				$this,
				'wc_am_api_key_text'
			), $this->wc_am_activation_tab_key );
			add_settings_field( $this->wc_am_api_key_key, esc_html__( 'API Key', $this->text_domain ), array(
				$this,
				'wc_am_api_key_field'
			), $this->wc_am_activation_tab_key, $this->wc_am_api_key_key );

			/**
			 * @since 2.3
			 */
			if ( $this->no_product_id ) {
				add_settings_field( 'product_id', esc_html__( 'License Type', $this->text_domain ), array(
					$this,
					'wc_am_product_id_field'
				), $this->wc_am_activation_tab_key, $this->wc_am_api_key_key );
			}

			add_settings_field( 'status', esc_html__( 'API Key Status', $this->text_domain ), array(
				$this,
				'wc_am_api_key_status'
			), $this->wc_am_activation_tab_key, $this->wc_am_api_key_key );
			// Activation settings
			register_setting( $this->wc_am_deactivate_checkbox_key, $this->wc_am_deactivate_checkbox_key, array(
				$this,
				'wc_am_license_key_deactivation'
			) );
			add_settings_section( 'deactivate_button', esc_html__( 'API Deactivation', $this->text_domain ), array(
				$this,
				'wc_am_deactivate_text'
			), $this->wc_am_deactivation_tab_key );
			add_settings_field( 'deactivate_button', esc_html__( 'Deactivate API Key', $this->text_domain ), array(
				$this,
				'wc_am_deactivate_textarea'
			), $this->wc_am_deactivation_tab_key, 'deactivate_button' );
		}

		/**
		 * Render "Product ID" field
		 * Overrides parent to change Product ID <input> field to <select> and add license type options
		 * @since 2.3
		 */
		public function wc_am_product_id_field() {
			$product_id = get_option( $this->wc_am_product_id );

			if ( ! empty( $product_id ) ) {
				$this->product_id = $product_id;
			}

			$license_types = apply_filters('cw_license_product_ids', [
				'wc_am_client_cryptowoo' => [
					'free' =>  13365, // Free version
					'single_store' => 1066, // Legacy single store
					'multi_store' => 1067, // Legacy multi store
					'premium' => 1068, // Legacy premium
					'perpetual_single' => 3528, // Legacy Perpetual single
					'perpetual_multi' => 3529, // Legacy Perpetual multi
					'perpetual_premium' => 3530, // Legacy Perpetual premium
				]
			]);

			if ( ! empty( $license_types[$this->data_key] ) ) {
				$select_options = sprintf('<select name="%1$s" id="%1$s">', $this->wc_am_product_id);
				foreach( $license_types[$this->data_key] as $type => $id) {
					$selected = (int)$product_id === (int)$id ? 'selected' : '';
					$type_string = ucfirst(str_replace('_', ' ', $type));
					$select_options .= sprintf('<option value="%1$s" %2$s>%3$s</option>', $id, $selected, $type_string);
				}
				$select_options .= '</select>';

				echo $select_options;
			} else {
				if ( ! empty( $product_id ) ) {
					echo "<input id='product_id' name='" . esc_attr( $this->wc_am_product_id ) . "' size='25' type='text' value='" . absint( $this->product_id ) . "' />";
				} else {
					echo "<input id='product_id' name='" . esc_attr( $this->wc_am_product_id ) . "' size='25' type='text' value='' />";
				}
			}
		}
	}
}
